;**********************************************************************************************************************************************************
;  FILENAME: SDCard.asm
;  Version: 2.14
;  DESCRIPTION: SDCard User Module software implementation file
;
;  (c) Ilya Mamontov 2008-2009, 2025 
;
;          SD, MMC  SDHC
;    
;     - SDCard_Run -  ,           32 .
;	      SDCard_Callback_Procedure. 
;    SDCard_Callback_Procedure      (32 )    LCD (   16 ).
;
;         (512 ),  SD  MMC     .
;         , ..     ""   (   ).
;	            "".
;   
;**********************************************************************************************************************************************************

include "m8c.inc"
include "memory.inc"
include "SDCard.inc"


;-----------------------------------------------
;  Macros Definitions
;-----------------------------------------------
macro SDCard_DelayMin ;a minimum setup time for the high/low transition
	; 3 MHz: 1 nop operation
	; 6 MHz: 2 of nop
	;12 MHz: 3 of nop
	;24 MHz: 6 of nop
	nop	
	nop	
	nop	
	nop	
	nop	
	nop	
endm
;-----------------------------------------------
;	Drive the SD Card pins
macro SDCard_CS0
	and reg[SDCard_CS_Port],~SDCard_CS_Mask	;Drive the CS pin low
	nop
	SDCard_DelayMin							;Wait a minimum setup time for the high/low transition
endm
;-----------------------------------------------
macro SDCard_CS1
	or reg[SDCard_CS_Port],SDCard_CS_Mask	;Drive the CS pin high
	nop
	SDCard_DelayMin							;Wait a minimum setup time for the high/low transition
endm
;-----------------------------------------------
macro SDCard_DI0
	and reg[SDCard_DI_Port],~SDCard_DI_Mask	;Drive the DI pin low
endm
;-----------------------------------------------
macro SDCard_DI1
	or reg[SDCard_DI_Port],SDCard_DI_Mask	;Drive the DI pin high
endm
;-----------------------------------------------
macro SDCard_CLK0
	and reg[SDCard_CLK_Port],~SDCard_CLK_Mask	;Drive the CLK pin low
endm
;-----------------------------------------------
macro SDCard_CLK1
	or reg[SDCard_CLK_Port],SDCard_CLK_Mask	;Drive the CLK pin high
endm
;-----------------------------------------------
macro SDCard_Cmd	;@0, @1, @2, @3, @4, @5
	SDCard_CS0						;Set SD Card select line low 
.wait:
	call SDCard_ReadByte
	cmp A, 0xFF
	jnz .wait

	mov A,@0
	call SDCard_SendByte
	mov A,@1
	call SDCard_SendByte
	mov A,@2
	call SDCard_SendByte
	mov A,@3
	call SDCard_SendByte
	mov A,@4
	call SDCard_SendByte
	mov A,@5
	call SDCard_SendByte
endm
;-----------------------------------------------
macro SDCard_CmdA	;@0, @1
	SDCard_CS0						;Set SD Card select line low 
.wait:
	call SDCard_ReadByte
	cmp A, 0xFF
	jnz .wait

	mov A,@0
	call SDCard_SendByte

	tst [etap],0x40					;   SD    
	;jmp .v2
	jz .v1

.v2:
	; SDHC
	mov A,[@1+3]
	call SDCard_SendByte
	mov A,[@1+2]
	call SDCard_SendByte
	mov A,[@1+1]
	call SDCard_SendByte
	mov A,[@1+0]
	call SDCard_SendByte
	jmp .s1

.v1:
	;; SD  MMC  2  - "@1"   512  
	mov A,[@1+1]
	asl A
	mov A,[@1+2]
	rlc A
	call SDCard_SendByte
	mov A,[@1+0]
	asl A
	mov A,[@1+1]
	rlc A
	call SDCard_SendByte
	mov A,[@1+0]
	asl A
	call SDCard_SendByte
	mov A,0
	call SDCard_SendByte
.s1:
	mov A,0xFF
	call SDCard_SendByte
endm
;-----------------------------------------------
macro SDCard_Cmd_00		;Sends the six byte reset command with 7 bit CRC of 0x95 (Byte String = 0x40 0x00 0x00 0x00 0x00 0x95) 
	call SDCard_Wait1ms				;Wait a minimum setup time for the high/low transition
	SDCard_CS0						;Set SD Card select line low 

.wait:
	call SDCard_ReadByte
	cmp A, 0xFF
	jnz .wait

	mov A,0x40
	call SDCard_SendByte
	mov A,0
	call SDCard_SendByte
	mov A,0
	call SDCard_SendByte
	mov A,0
	call SDCard_SendByte
	mov A,0
	call SDCard_SendByte
	mov A,0x95
	call SDCard_SendByte
endm
;-----------------------------------------------
macro SDCard_CmdEnd
	mov A,0xFF						;Send a null byte (Min 8 clock cycles required)	
	call SDCard_SendByte
	SDCard_CS1						;Set SD Card select line high (disables card for SPI)
endm
;-----------------------------------------------
macro SDCard_GetR1	;Gets R1 command response
	mov A,0xFF
	call SDCard_SendByte
	call SDCard_ReadByte	;;result in A:   bits 0=Idle, 1=Erase Reset, 2=Illegal CMD, 3=CRC Error, 4=Erase_Seq_Error, 5=Address Error, 6=Param Error
endm
;-----------------------------------------------
macro SDCard_GetR2	;Gets R2 command response
	mov A,0xFF
	call SDCard_SendByte
	call SDCard_ReadByte
	call SDCard_ReadByte	;result in A
endm
;-----------------------------------------------
macro SDCard_GetR3	;Gets R3 command response (  OCR,  30(CCS) = 1  SDHC  = 0  SD)
	mov A,0xFF
	call SDCard_SendByte
	call SDCard_ReadByte	;"R1"
	call SDCard_ReadByte	;.
	mov X,A	; 6 - "CCS"
	call SDCard_ReadByte	;
	call SDCard_ReadByte	;
	call SDCard_ReadByte	;.
endm
;-----------------------------------------------
macro SDCard_SkipM	@0	;Read @0 dummy bytes
	mov [cnt],@0
	call SDCard_Skip
endm
;-----------------------------------------------
macro SDCard_ReadBuffM	@0,@1	;Read @0 bytes into buffer @1
	mov [cnt],@0
	mov X,@1
	call SDCard_ReadBuff
endm
;-----------------------------------------------
macro SDCard_Move	@0,@1	;Copy @0 <- @1
	mov [@0+0],[@1+0]
	mov [@0+1],[@1+1]
	mov [@0+2],[@1+2]
	mov [@0+3],[@1+3]
endm
;-----------------------------------------------
macro SDCard_Add	@0,@1	;@0 = @0 + @1
	mov A,[@1+0]					;DataA =DataA + DataAddr 
	add   [@0+0],A
	mov A,[@1+1]
	adc   [@0+1],A
	mov A,[@1+2]
	adc   [@0+2],A
	mov A,[@1+3]
	adc   [@0+3],A
endm
;-----------------------------------------------
macro SDCard_mul @0,@1,@2	;Unsigned multiplication @0(4) = @1(2) * @2(1)
	mov [@0+0],0	;clear  result
	mov [@0+1],0
	mov [@0+2],0
	mov [@0+3],0

	mov [tmp],[@2]	;copy @2 -> temp
	mov [cnt],8		;load count
.mul_1:
	asl [@0+0]		;shift left @0
	rlc [@0+1]
	rlc [@0+2]
	rlc [@0+3]

	rlc [tmp]		;shift left tmp (@2)
	jnc .mul_2		;C=0

	mov A,[@1+0]	;else @0 = @0 + @1
	add   [@0+0],A
	mov A,[@1+1]
	adc   [@0+1],A
	adc   [@0+2],0
	adc   [@0+3],0
.mul_2:
	dec [cnt]			
	jnz .mul_1		;repeat till 8 bits are done
endm
;-----------------------------------------------



;-----------------------------------------------
; Variable Allocation
;-----------------------------------------------
area SDCard_RAM (RAM, REL, CON)
;------------ work variables ------------
etap::			BLK	1   ;    SD 
tmp:			BLK	2
cnt::			BLK	2	;    (.)     16  (  )
cnt8:			BLK	1	;counter of bits for SPI
cntMax::		BLK	2	;counter of dir entries
ptr::			BLK	2	;pointer to current char     (for _FindFile procedures)
ptrA:			BLK	1 	;pointer to alternative char (for _FindFile procedures) -        "?" !
altName::		BLK	4 	;(max) alternative char array(for _FindFile procedures) -        "?" !
dcnt::		BLK	1;    


buff::			BLK	BUFFERSIZE*BUFFNUMBER 	;data buffer (size is defind in SDCard.inc)
;---------- work FAT16 variables ----------
FClst::			BLK	2	;File start cluster
FSize::			BLK	4	;File size (long)
DataA:			BLK	4	;Address of current DATA block
ClstN::			BLK	2	;Current cluster number

;------------ FAT16 parameters ------------
Fat_LBA::				BLK	4	;Address of FAT area
Dir_LBA::				BLK	4	;Address of DIR area
Data_LBA::				BLK	4	;Address of DATA area
Sectors_per_cluster:	BLK	1	;Number of sectors per cluster 1...64 (Usually one)
N_of_dir_enries::		BLK	2	;Number of root directory enries      (Usually 512)
FSizeCnt::						;(4) work variable
N_of_FAT_tables:		BLK	1	;Number of FAT tables                 (Standard 2)
N_of_sectors_per_FAT:	BLK	3	;(2) Number of sectors per FAT 1..64K (Depends on disk size)

area UserModules (ROM, REL)

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_SendByte
;       Sends a byte (using SPI protocol)
;  ARGUMENTS:
;     A - data to transmit.
;  RETURNS:  none
;-----------------------------------------------------------------------------
.SECTION
SDCard_SendByte:
  	mov [cnt8],8
.lp8:
	rlc A
	jc .c1
		SDCard_DI0
		SDCard_CLK0
		jmp .cnext
	.c1:SDCard_DI1
		SDCard_CLK0
		jmp .cnext
	.cnext:
		nop
	SDCard_CLK1
	dec [cnt8]
	jnz .lp8
	SDCard_CLK0
  	ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_ReadByte
;       Receives a byte (using SPI protocol)
;  ARGUMENTS: none
;  RETURNS:  A - received data
;-----------------------------------------------------------------------------
.SECTION
SDCard_ReadByte:
  	mov [cnt8],8
.loop8:
	SDCard_CLK1
	tst reg[SDCard_DO_Port],SDCard_DO_Mask	;Check the DO pin
	jz .z0
		or  F,4		;C=1
		jmp .znext
	.z0:and F,~4	;C=0
		jmp .znext
	.znext:
	rlc A
	SDCard_CLK0
	dec [cnt8]
	jnz .loop8
  	ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_Skip
;       Reads N dummy bytes
;  ARGUMENTS:
;     [cnt] - number of bytes to read 1...255 (0=256)
;  RETURNS:  none
;-----------------------------------------------------------------------------
.SECTION
SDCard_Skip::
.loop:
	call SDCard_ReadByte
	dec [cnt]
	jnz .loop
	ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_ReadBuff
;       Reads N bytes from SD Card to buffer [X]
;  ARGUMENTS:
;     X - address of buffer
;     [cnt] - number of bytes
;  RETURNS:  none
;-----------------------------------------------------------------------------
.SECTION
SDCard_ReadBuff:
.loop:
	call SDCard_ReadByte
	mov [X],A
	inc X
	dec [cnt]
	jnz .loop
	ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_CheckReply
;       Waits for reply byte 0xFE
;  ARGUMENTS:
;  RETURNS:  zero (Z=1) if success
;            nz   (Z=0) if card failed 
;-----------------------------------------------------------------------------
.SECTION
SDCard_CheckReplyFE:
	mov [cnt],254				;Set number of tries until you timeout
.wait:
	call SDCard_ReadByte
	cmp A,0xFE
	jz .endwait					;Loop until status is good (Z=1) , or
	dec [cnt]
	jnc .wait					;until a timeout occurs
	;Z=0 
.endwait:
	ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_Wait1ms
;       Delay 1ms for any clock frequency
;       A is modified!
;-----------------------------------------------------------------------------
.SECTION
SDCard_Wait1ms:
	M8C_SetBank1
	mov A,reg[OSC_CR0]			;get CPU clock
	M8C_SetBank0
	and A,07h 					;mask off only the clock bits
	index SDCard_Delay_Table	;get delay value
	mov [cnt],A
	.loop:				;x 10 us
		nop
		mov [cnt8],18
		.loop10us:
			dec [cnt8]
		jnz .loop10us
	dec [cnt]
	jnz .loop
  	ret
.ENDSECTION
.LITERAL
SDCard_Delay_Table: DB   13,  25,   50,  100,    7,     4,    2,    1
;---------------------- 3MHz 6MHz 12MHz 24MHz 1.5MHz 750kHz 188kHz 94kHz
.ENDLITERAL


;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_InitCard
;       Initializes SD card and gets FAT16 information
;  RETURNS: [etap]=
;			=0x*0 - ,  4  -  :
;						0x00 -   
;						0x10 - SD  2   (v1.0)
;						0x20 - MMC  2  (v1.0)
;						0x30 - SD  2   (v2.0)
;						0x40 - SDHC ( 2 ) v2.0
;			=0x01 -      SPI ( )
;			=0x02 -     SPI,   V1.0 (SD/MMC),    
;			=0x03 -     SPI,   V2.0 (SD/SDHC),    
;			=0x04 -     SPI,   V2.0,     SD-SDHC
;			=0x*5 -   ,      (  0- ,    )  ( 4  -  )
;			=0x*6 -  FAT16 - 0   
;			=0x*7 -  FAT16 -  2  FAT
;			=0x*8 -  FAT16 -   "FAT16"

;    A: 0=Success, 1=SD card failed, 2=FAT type is missing
;         Array of FAT16 parameters:
;    [FatAddr]				(4) - Address of FAT area
;    [DirAddr]				(4) - Address of DIR area
;    [DataAddr]				(4) - Address of DATA area
;    [Sectors_per_cluster]	(1) - Number of sectors per cluster 1...64 (Usually one)
;    [N_of_dir_enries]		(2) - Number of root directory enries     (Usually 512)
;    [N_of_FAT_tables]		(1) - Number of FAT tables               (Standard 2)
;    [N_of_sectors_per_FAT]	(2) - Number of sectors per FAT 1..64K  (Depends on disk size)
;-----------------------------------------------------------------------------
.SECTION
 SDCard_InitCard::
_SDCard_InitCard::
	mov [etap],0x01			;   
	SDCard_CS1
    ;If CPU clock = 24 MHz then set 12 MHz for use under 400Khz (Must run slowly until card initialized as SD or MMC)
    M8C_SetBank1
	mov A,reg[OSC_CR0]
	mov [tmp],A				;save current value
	and A,0x07				;mask off only the clock bits
	cmp A,3					;clock=24 MHz?
	jnz skip_12MHz			;no
		and reg[OSC_CR0],~1	;clear bit 0 (24 -> 12 MHz)
	skip_12MHz:
    M8C_SetBank0
   	;Wait at least 1 ms after power up before attempting to do anything with the card
	call SDCard_Wait1ms
	call SDCard_Wait1ms		;  -   1 
	;Send out a string of at least 74 clocked highs (10 or more 0xFF characters) previous to sending commands
	;mov [cnt],10
	mov [cnt],20		;  -    20  0xFF,   10 
	.loop:
		mov A,0xFF ;FF
		call SDCard_SendByte
		dec [cnt]
	jnz .loop  

	;Set card to SPI mode - Set SD Card pin 1 low and then send CMD(0) Reset
	;If the CS line is low then a mode R1 SPI response is generated.
	SDCard_Cmd_00
	SDCard_GetR1					;result in A:     bits 0=Idle, 1=Erase Reset, 2=Illegal CMD, 3=CRC Error, 4=Erase_Seq_Error, 5=Address Error, 6=Param Error
	  cmp A,0x1
	  jnz noCARD					;etap=0x01 (     SPI)
	SDCard_CmdEnd

	;    SPI
	inc[etap]						;etap=0x02
	
	;  Ver1.0/Ver2.0  (8 GB  FF FF)
	SDCard_Cmd 0x48,0,0,0x01,0xAA,(0x43<<1)|1    ;CMD8,  CRC! R1=0x01/0x05
	SDCard_GetR1					;result in A:  R1=0x01/0x05    bits 0=Idle, 1=Erase Reset, 2=Illegal CMD, 3=CRC Error, 4=Erase_Seq_Error, 5=Address Error, 6=Param Error
	  and A,0x4
	  jnz .testV1 ; "CMD8"   -    Ver1.0    
	
	; Ver2.0 (8 GB  FF FF)
	inc[etap]						;etap=0x03
	SDCard_GetR1					;result in A:
	SDCard_GetR1					;result in A:	
	SDCard_GetR1					;result in A: bit 0 must be 1	
	and A,1
	  ;jz noCARD	
	SDCard_GetR1					;result in A: must be 0xAA	
	  cmp A,0xAA	
	  ;jnz noCARD		
	SDCard_CmdEnd
	
	; 
	mov [cnt+0],255					;   = 767 (255  , 522 -    512  )
	mov [cnt+1],2					;
.looptestV2:
	SDCard_Cmd 0x77,0,0,0,0,0xFF;   ;Send CMD55
	SDCard_GetR1					;result in A:     bits 0=Idle, 1=Erase Reset, 2=Illegal CMD, 3=CRC Error, 4=Erase_Seq_Error, 5=Address Error, 6=Param Error
	SDCard_CmdEnd

	SDCard_Cmd 0x69,0x40,0,0,0,0xFF	;Send ACMD41 command string (REQUIRES a CMD55 before)
	SDCard_GetR1					;result in A:     bits 0=Idle, 1=Erase Reset, 2=Illegal CMD, 3=CRC Error, 4=Erase_Seq_Error, 5=Address Error, 6=Param Error
	mov X,A	;save result
	SDCard_CmdEnd

	dec [cnt+0]
	sbb [cnt+1],0
	jc noCARD						; 255 ,        "CMD55-ACMD41" ( ) - ,   
	;jc .noCARDDD
	mov A,X	;restore result
	jnz .looptestV2					; 

	; Ver2.0 (SD  SDHC)  
	inc[etap]						;etap=0x04 (   SD/SDHC)
	;,  -   CMD58 ( R3  -   OCR,  30(CCS) = 1  SDHC  = 0  SD)
	SDCard_Cmd 0x7A,0,0,0,0,0xFF;   ;Send CMD58
	;SDCard_GetR3					;  R3 -  CCS    6  X
	;  R3.    : R1    4  
	SDCard_GetR1					;result in A:     bits 0=Idle, 1=Erase Reset, 2=Illegal CMD, 3=CRC Error, 4=Erase_Seq_Error, 5=Address Error, 6=Param Error
	and A,0xFF
	jnz noCARD						;  SD/SDHC
	call SDCard_ReadByte	;.  OCR
	mov X,A	; 6 - "CCS"
	call SDCard_ReadByte	;
	call SDCard_ReadByte	;
	call SDCard_ReadByte	;.  OCR
	SDCard_CmdEnd

	mov [etap],0x35					;etap=0x35 (      SD v2.0 <2 )
	mov A,X
	and A,0x40
	jz .noSDHC
		add [etap],0x10				;etap=0x45 (   SDHC >2 )		
	.noSDHC:
	jmp continue2	
	
.testV1:
	;  Ver1.0 (SD  MMC),  ,   
	mov [cnt],255					; 
.looptestV1:
	SDCard_Cmd 0x77,0,0,0,0,0xFF	;Send CMD55 command string (next command is an ACMD)
	SDCard_GetR1
	SDCard_CmdEnd

	SDCard_Cmd 0x69,0,0,0,0,0xFF	;Send ACMD41 command string (REQUIRES a CMD55 before)
	SDCard_GetR1
	mov X,A	;save result

	and A,4						;   CMD41 ?
	jz .CMD41_supported			;,    "CMD55-CMD41"
		;,  CMD55-CMD41    CMD1  
		.loop_CMD1:
			SDCard_Cmd 0x41,0,0,0,0,0xFF	;Send CMD1 command string
			SDCard_GetR1
			mov X,A	;save result
			SDCard_CmdEnd
			dec [cnt]
			jz noCARD						; 255 ,        "CMD1" ( ) - ,   
			mov A,X	;restore result
			jnz .loop_CMD1					; 
			
			;    "CMD1" ( MMC Ver1.0  2 )
			mov [etap],0x25					;etap=0x25
			jmp continue2	
		
	.CMD41_supported:
	SDCard_CmdEnd
	dec [cnt]
	jz noCARD						; 255 ,        "CMD55-CMD41" ( ) - ,   
	mov A,X	;restore result
	jnz .looptestV1					; 
	;   " CMD55-ACMD41" ( SD Ver1.0  2 )
	mov [etap],0x15					;etap=0x15

   
continue2:		;=============     ,    
;include "to_LCD.inc"
			; 
			;toLCD_String 0,0,"continue2="	;debug!!
			;mov A,[etap]
			;call LCD_PrHexByte
			;halt


	;Set CPU clock back to original settings used in project
    M8C_SetBank1
	mov A,[tmp]
	mov reg[OSC_CR0],A
    M8C_SetBank0

	
	;  Data_LBA
	mov [Data_LBA+0],0
	mov [Data_LBA+1],0
	mov [Data_LBA+2],0
	mov [Data_LBA+3],0
	; 0-  (LBA=0)
	;SDCard_Cmd 0x51,0,0,0x01,0xC6,0xFF	;CMD17 - Read single block command (  =454)
	SDCard_Cmd 0x51,0,0,0x00,0x00,0xFF	;CMD17 - Read single block (512) command (  =0.    SD  SDHC.) 
	SDCard_GetR1
	call SDCard_CheckReplyFE
	jnz noCARD						;etap=0x*5 (  0- )

	;SDCard_SkipM 200				; 

	;  0-  - PBR (   11  12  0  2 )  MBR (  ).
	;( 1-        )
	SDCard_SkipM 11			;  11 
	call SDCard_ReadByte
	mov X,A					; 
	call SDCard_ReadByte
	cmp A,2					;   12  2 ?
	jnz .readMBR
		mov A,X				;   11  0 ?
		jz .readDiskInfo	;   -  0-   PBR        .

.readMBR:
	; 0-  -    (MBR), 13   ,   1-  (   )
	SDCard_SkipM 187			;  187  (=200-13)
	SDCard_SkipM 254			; ,     454 (      254 )
	SDCard_ReadBuffM 4,Data_LBA	; 4   1-  (Boot Sector LBA -  )
	SDCard_SkipM 54+2			;  + 2  CRC ( )
	SDCard_CmdEnd

	;   PBR (  ) -  ,   Data_LBA
	;SDCard_Cmd 0x51,0,0,0x00,0x00,0xFF	;CMD17 - Read single block (512) command (  =0.   SD  SDHC.) 
	SDCard_CmdA 0x51,Data_LBA		;Read block command
;	SDCard_Cmd 0x51,0,0,0x02,0x00,0xFF	;CMD17 - Read single block (512) command (  =0.    SD  SDHC.) 
	SDCard_GetR1
	call SDCard_CheckReplyFE
	jnz noCARD						;etap=0x*5  ("   ")

	;    ( PBR)
	inc[etap]					;etap=0x*6  (" 512   ")
	SDCard_SkipM 11				;  11  
	call SDCard_ReadByte		;. 2  - -   ,   512
	cmp A,0
jnz notFAT					; 512  
	call SDCard_ReadByte
	cmp A,2
jnz notFAT					; 512  

.readDiskInfo:
	;     (13   , -    )

	; -   
	call SDCard_ReadByte
	mov [Sectors_per_cluster],A
	; -  
	call SDCard_ReadByte
	mov [Fat_LBA+0],A
	call SDCard_ReadByte
	mov [Fat_LBA+1],A
	mov [Fat_LBA+2],0
	mov [Fat_LBA+3],0
	; Fat_LBA  -   + Boot Sector LBA
	SDCard_Add  Fat_LBA,Data_LBA
	
	; - FAT (=2)
	inc[etap]						;etap=0x*7
	call SDCard_ReadByte
	mov [N_of_FAT_tables],A
	cmp A,2	;  2  FAT
jnz notFAT
	;get number of root directory enries
	call SDCard_ReadByte
	mov [N_of_dir_enries+0],A
	call SDCard_ReadByte
	mov [N_of_dir_enries+1],A
	SDCard_SkipM 3						;skip bytes
	
	; Dir_LBA (  ) = FAT_LBA + sectors per FAT * # of FATs
	;get number of sectors per FAT
	call SDCard_ReadByte
	mov [N_of_sectors_per_FAT+0],A
	call SDCard_ReadByte
	mov [N_of_sectors_per_FAT+1],A
	mov [N_of_sectors_per_FAT+2],0
	SDCard_Move Dir_LBA,Fat_LBA
	loop_calc:
		mov A,[N_of_sectors_per_FAT+0]
		add [Dir_LBA+0],A
		mov A,[N_of_sectors_per_FAT+1]
		adc [Dir_LBA+1],A
		mov A,[N_of_sectors_per_FAT+2]
		adc [Dir_LBA+2],A
		dec [N_of_FAT_tables]
		jnz loop_calc
	
	; Data_LBA (  ) = Dir_LBA + number of root directory enries ( 512) * 32 / 512 (  )
	;                                         = Dir_LBA + number of root directory enries ( 512) / 16
	mov [Data_LBA+0],[N_of_dir_enries+0]
	mov [Data_LBA+1],[N_of_dir_enries+1]
	mov [Data_LBA+2],0
	mov [Data_LBA+3],0
	mov [cnt],4
	.loop_div16:
		asr [Data_LBA+3]
		rrc [Data_LBA+2]
		rrc [Data_LBA+1]
		rrc [Data_LBA+0]
		dec [cnt]
		jnz .loop_div16
	SDCard_Add  Data_LBA,Dir_LBA	;+ Dir_LBA
	
	;  FAT  "FAT16"
	SDCard_SkipM 30					;skip bytes
	inc[etap]						;etap=0x*8
	call SDCard_ReadByte
	cmp A,'F'
jnz notFAT
	call SDCard_ReadByte
	cmp A,'A'
jnz notFAT
	call SDCard_ReadByte
	cmp A,'T'
jnz notFAT
	call SDCard_ReadByte			; 58- 
	cmp A,'1'
jnz notFAT
	call SDCard_ReadByte			;  59- 
	cmp A,'6'
jnz notFAT
	mov [tmp],0
	SDCard_SkipM 200				;
	SDCard_SkipM 253+2				; + 2  CRC ( )

	; 4 
	;toLCD_String 0,0,"OK4    "	;debug!!
	;toLCD_PrHexInt cnt              ;  
	;halt                          ;



	and [etap],0xF0						;etap=0x*0
	jmp InitCard_end		;Success!

noCARD:
	mov [tmp],1				;if card failed
	jmp InitCard_end
notFAT:
	mov [tmp],2				;if FAT type is missing
InitCard_end:
	;SDCard_CmdEnd			;finish the command cycle
	SDCard_CS1				;disables card
	mov A,[tmp]				;return code

  	ret
.ENDSECTION



;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_FindFile_in_Root
;       Finds a file (or subdir) in the root directory.
;  ARGUMENTS:
;       [ptr]     (2)- address of string 8+3+1 bytes
;       [Dir_LBA] (2)- start LBA of directory 
;  RETURNS:
;		[etap]: 0x*0=success, 0x*F=no card,   = "  " 
;       [FClst] (2) - file start cluster
;       [FSize] (4) - file size
;  Features:
;       1. '*' seeks any char, '?' seeks with char located in altName buffer. 
;       2. 12-th char is an attribute byte: 0x00=file, 0x10=directory, 0x08=label  (Attribute bits: 0=Read Only, 1=Hidden, 2=System, 3=Volume Label, 4=SubDir, 5=Archive)
;-----------------------------------------------------------------------------
.SECTION
 SDCard_FindFile_in_Root::
_SDCard_FindFile_in_Root::
	mov [cntMax+0],[N_of_dir_enries+0]	;load number of entries in root dir ( :   32 )
	mov [cntMax+1],[N_of_dir_enries+1]

	;mov [cntMax+0],0x2	;load number of entries in root dir ( :   32 )
	;mov [cntMax+1],0x0

	SDCard_Move DataA,Dir_LBA			;initialize DataA pointer (  LBA)

	;SDCard_CmdA 0x51,DataA			;Read block command (512 )
	;call SDCard_CheckReplyFE
	;toLCD_Position  0, 0
	;call SDCard_ReadByte			;read byte
	;call LCD_PrHexByte
	;call SDCard_ReadByte			;read byte
	;call LCD_PrHexByte
	;call SDCard_ReadByte			;read byte
	;call LCD_PrHexByte
	;call SDCard_ReadByte			;read byte
	;call LCD_PrHexByte
	;halt                          ;
	jmp SDCard_FindFile_Entry_point

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_FindFile_in_Cluster
;       Finds a file in cluster.
;  ARGUMENTS:
;       [ptr] (2) - address of string 8+3+1 bytes
;       [ClstN](2)- cluster number
;  RETURNS:
;		[etap]: 0x*0=success, 0x*F=no card,   = "  " 
;       [FClst] (2) - file start cluster
;       [FSize] (4) - file size
;  Features:
;       1. '*' seeks any char, '?' seeks with char located in altName buffer. 
;       2. 12-th char is an attribute byte: 0x00=file, 0x10=directory, 0x08=label  (Attribute bits: 0=Read Only, 1=Hidden, 2=System, 3=Volume Label, 4=SubDir, 5=Archive)
;-----------------------------------------------------------------------------
 SDCard_FindFile_in_Cluster::
_SDCard_FindFile_in_Cluster::
	;load number of entries per cluster (= sect per cluster * 16)
	mov [cntMax+0],[Sectors_per_cluster]
	mov [cntMax+1],0
	mov A,4
	.loop4:
		asl [cntMax+0]
		rlc [cntMax+1]
		dec A
	jnz .loop4	
	call SDCard_Cluster_to_LBA		;cluster number -> LBA of data

	
;-----Entry_point-------->
SDCard_FindFile_Entry_point:
	;SDCard_Cmd 0x50,0,0,0,32,0xFF	;CMD16 - Set block size command
	;SDCard_GetR1
	;SDCard_CmdEnd

	;mov [DataA+0],[DataA+1]		;  SDHC        ( *512),   LBA (..   512) 
	;mov [DataA+1],[DataA+2]
	;mov [DataA+2],[DataA+3]
	;mov [DataA+3],0
	;add A,0 ;C=0
	;rrc [DataA+2]
	;rrc [DataA+1]
	;rrc [DataA+0]

.loop_sectors:

	SDCard_CmdA 0x51,DataA			;Read block command (512 )
	SDCard_GetR1
	call SDCard_CheckReplyFE
	jnz .noCARD						;[etap]=0x*9    

.loop_within_sector:

	mov [cnt],12					;initialize counter - lenght of name + ext + attribute byte
	mov X,[ptr+0]					;initialize pointer to name in ROM - [tmp+1]:X
	mov [tmp+1],[ptr+1]
	mov [ptrA],altName				;initialize pointer to alternative chars in RAM

.loop12: ;11   + 1  
	;read char from SD
	call SDCard_ReadByte			;read byte
	mov [tmp],A						;save current char from SD to [tmp+0]
	;load current char from ROM using pointer [tmp+1]:X
	mov A,[tmp+1]
	romx
	cmp [cnt],1						;  (.. )?
	jnz .skip_att					;no
		and [tmp],0x18				;    
	.skip_att:

	cmp [cnt],12					;  (   )?
	jnz .skip_check					;no
		cmp [tmp],0x00				;record is blank?
		jz .not_found				;    -  
		;cmp [tmp],0xFF				;record is blank?
		;jz .not_found				;yes, that's directory end and file is not found 
	.skip_check:
	
	cmp A,'*'						;is "any char"?
	jz .skip_cmp
		cmp A,'?'					;is "alternative char"?
		jnz .skip_alt
			mvi A,[ptrA]			;load char from altName array
		.skip_alt:
		or A,0x20					;make char from ROM lowercase
		or [tmp],0x20				;make char from SD lowercase
		cmp A,[tmp]					;compare chars
		jnz .next_record			;not equal
	.skip_cmp:

	inc X							;increment pointer
	adc [tmp+1],0
	dec [cnt]						;do 12 times
	jnz .loop12
	
	;now file is found, get other information
	SDCard_SkipM 14
	SDCard_ReadBuffM 2,FClst		;file start cluster
	SDCard_ReadBuffM 4,FSize		;file size.      32  .
	;SDCard_CmdEnd
	dec [cntMax+0]					;   Dir      
	sbb [cntMax+1],0
	and [etap],0xF0					;[etap]=0x*0 ()
	jmp .end
	
.next_record:
	add [cnt],19					;    (  32 )
	call SDCard_Skip				;
	;SDCard_CmdEnd					; -  SDHC   SDCard_CmdEnd, ..      512
	;next record address		; SDHC  ,      
	;add [DataA+0],32
	;adc [DataA+1],0
	;adc [DataA+2],0
	;adc [DataA+3],0
	
	;   Dir
	dec [cntMax+0]
	sbb [cntMax+1],0
	jnz .skip
	mov A,[cntMax+0]
	jz .not_found2					;  (=0) -    
	.skip:
	
	;   16-    ()
	tst [cntMax+0],0xF				;  4   0
	jnz .loop_within_sector			;    -     
	    ;   -     (LBA),     ,    
		call SDCard_ReadByte		;  2  CRC
		call SDCard_ReadByte		;  2  CRC
		add [DataA+0],1
		adc [DataA+1],0
		adc [DataA+2],0
		adc [DataA+3],0
		SDCard_CmdEnd
		jmp .loop_sectors
	
.not_found:  ;   - ..  
.not_found2: ; ,      
	jmp .end						;[etap]    (     "  ")
.noCARD:
	or[etap],0x0F					;[etap]=0x*F  (  )
.end:
	;    (   ) -     ! 
	and [cntMax+0],0xF				;     4  (16   )
	jz .skip2						;    -   
		.rep2:
			SDCard_SkipM 32
			dec [cntMax+0]
			jnz .rep2
	.skip2:
	call SDCard_ReadByte		;  2  CRC
	call SDCard_ReadByte		;  2  CRC
	SDCard_CmdEnd
 	ret
.ENDSECTION


;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_Cluster_to_LBA
;       Converts cluster number -> LBA of data
;                       [ClstN] -> [DataA]
;  ARGUMENTS:
;       [ClstN] (2)- cluster number
;  RETURNS:
;       [DataA] (4) - LBA of data
;-----------------------------------------------------------------------------
.SECTION
SDCard_Cluster_to_LBA::
	; LBA   = Data_LBA + (cluster number - 2) * sectors_per_cluster
	sub [ClstN+0],2							        ; ClstN = ClstN -2 
	sbb [ClstN+1],0
	SDCard_mul	DataA,ClstN,Sectors_per_cluster		;DataA = ClstN * Sectors_per_cluster
	SDCard_Add  DataA,Data_LBA				        ;        + Data_LBA
	add [ClstN+0],2							        ; ClstN
	adc [ClstN+1],0
  	ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_NextCluster
;       Reads a data in FAT area to define next cluster in chain
;  ARGUMENTS:
;       [ClstN] (2) - cluster number, is replaced by new
;  RETURNS:
;		[ClstN] (2) - next cluster 
;		[etap]=0x*F -   ,  
;		A=1 -   , A=0 -  
;  DISCARDS: [tmp+0], [tmp+1]
;-----------------------------------------------------------------------------
.SECTION
SDCard_NextCluster::
	;define next cluster (2 bytes from address: FAT area address + (cluster number) * 2)   SDHC    2 !

	;      LBA    LBA ()  
	mov [tmp],[ClstN+0] 			;[tmp] (2) =0...255 -   LBA (    , ..     2)
	mov [DataA+0],[ClstN+1] 		;[DataA] (4) -   LBA 
	mov [DataA+1],0
	mov [DataA+2],0
	mov [DataA+3],0
	SDCard_Add	DataA, Fat_LBA		; DataA    Fat_LBA

	SDCard_CmdA 0x51,DataA			;Read block command
	SDCard_GetR1
	call SDCard_CheckReplyFE
	jnz .noCARD						;check result
	
	; ,       (  ) 
	mov A,255
	sub A,[ClstN+0]
	mov [tmp+1],A

	;         LBA ()
.loop:	
	call SDCard_ReadByte			;  2   
	mov [ClstN+0],A
	call SDCard_ReadByte
	mov [ClstN+1],A
	dec [tmp]
	jnc .loop
	;, [ClstN] (2) -  . 
	
	;    (   ) -     ! 
	.rep:
		call SDCard_ReadByte
		call SDCard_ReadByte
		dec [tmp+1]
		jnc .rep				;  1   (..  2  CRC)
	SDCard_CmdEnd
	mov A,0							;return code ""
  	ret

.noCARD:
	SDCard_CmdEnd
	or [etap],0x0F
	mov A,1							;return code " "
	ret
.ENDSECTION
;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_ReadDataBuff
;           (  buf  32 ),
;         buf  -   SDCard_Callback_Procedure,   A  -   (=32 ,    - )
;  ARGUMENTS:
;       [FClst] (2)-   ,    [ClstN] (2) ( )
;  RETURNS:
;		A: 0=success, 1=no card
;-----------------------------------------------------------------------------
.SECTION
SDCard_ReadDataBuff::
	mov [ClstN+0],[FClst+0]				;   
	mov [ClstN+1],[FClst+1]
	SDCard_Move FSizeCnt,FSize			;  

;    
.loop:	
	call SDCard_Cluster_to_LBA		;     LBA
	;SDCard_Cmd 0x50,0,0,0,BUFFERSIZE,0xFF	;CMD16 - Set block size command
	;SDCard_GetR1
	;SDCard_CmdEnd

	mov [tmp+1],[Sectors_per_cluster]	;    


	;toLCD_Position  0, 0	;
	;mov A, [tmp+1]
	;call LCD_PrHexByte
	;SDCard_CmdEnd
	;halt;


;    
.loop1:
	mov [tmp+0],(512/BUFFERSIZE)		;   512/32 = 16
	;mov [tmp+0],16		;  

	SDCard_CmdA 0x51,DataA			;Read block command ( 512 )
	SDCard_GetR1
	call SDCard_CheckReplyFE
	;jmp .noCARD	;
	jnz .noCARD	;  
	
;    
.loop2:
	mov A,[FSizeCnt+0]				;   ?
	or  A,[FSizeCnt+1]
	or  A,[FSizeCnt+2]
	or  A,[FSizeCnt+3]
	jz .end							;,  

	sub [FSizeCnt+0],BUFFERSIZE		;       (32) 
	sbb [FSizeCnt+1],0
	sbb [FSizeCnt+2],0
	sbb [FSizeCnt+3],0
 
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	;    (32 ) -   SDCard_ReadBuffM BUFFERSIZE,(buff+0)
	mov [cnt],BUFFERSIZE
	mov X,[ptrW]					;load pointer
	call SDCard_ReadBuff
	;SDCard_CmdEnd
	SDCard_CS1			;    -   ,     

	;  buff 
		;   <0,   -    A   (=[FSizeCnt+0]+BUFFERSIZE),  A = 32 -  
		mov A,[FSizeCnt+0]
		add A,BUFFERSIZE
		tst [FSizeCnt+3],0x80		;   ( 7 - )
		jnz .skip					; ""
			mov A,BUFFERSIZE		; "" -  A=32
		.skip:
	;
	; call SDCard_Callback_Procedure     (  )
	;
	;  -   ptrW	 32  ( )   ,   
	add [ptrW],A;BUFFERSIZE
	cmp [ptrW],buff+BUFFERSIZE*BUFFNUMBER
	jc .skip_init
		sub [ptrW],BUFFERSIZE*BUFFNUMBER		
	.skip_init:
	;     (       32+1,  )
	.wait_free_space:
		mov A,[ptrR]
		sub A,[ptrW]
		jnc .skip_corr
			add A,BUFFERSIZE*BUFFNUMBER
		.skip_corr:
		cmp A,BUFFERSIZE+1
		jc .wait_free_space
	SDCard_CS0			;   
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	tst [FSizeCnt+3],0x80		; 
	jnz .end					; "" (  ,   ,  ) - 

	dec [tmp+0]
	jnz .loop2		;  16  (  )

	call SDCard_ReadByte	;  -    2  CRC
	call SDCard_ReadByte

	;toLCD_Position  0, 0
	;mov A, [tmp+1]
	;call LCD_PrHexByte
	;SDCard_CmdEnd
	;halt;

	add [DataA+0],1				; LBA ( )
	adc [DataA+1],0
	adc [DataA+2],0
	adc [DataA+3],0

	dec [tmp+1]
	jnz .loop1					;      

	call SDCard_NextCluster	;   
	cmp A,1
	jz .noCARD	;check result
	;    ""  "" FFF7..FFFF
	cmp [ClstN+1],0xFF
	jnz .loop		;    / ( ClstN   FF**,  . -  )
	cmp [ClstN+0],0xF7
	jc  .loop		;    / (ClstN < FFF7)

.end:
	;    -     (   ) -     ! 
	;[tmp+0] -   
	asl [tmp+0]		;  () =   * 16 + 2( CRC).    , ..     (  ). 
	asl [tmp+0] 
	asl [tmp+0] 
	asl [tmp+0]
	add [tmp+0],2 
.loop3:
	call SDCard_ReadByte
	dec [tmp+0]
	jnz .loop3		;  
	SDCard_CmdEnd
	mov A,0			;return code ""
	ret
.noCARD:
	SDCard_CmdEnd
	or [etap],0x0F
	mov A,1			;return code " "
	ret
.ENDSECTION


;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_Run
;       Run all procedures to read file (using path, name from ROM + alternative chars from RAM)
;  ARGUMENTS:
;       see code
;  RETURNS:
;		A: 0=success, 1=no card, 2=file not found
;       [etap]=
;			=0x*0 - ,  4  -  :
;						0x00 -   
;						0x10 - SD  2   (v1.0)
;						0x20 - MMC  2  (v1.0)
;						0x30 - SD  2   (v2.0)
;						0x40 - SDHC ( 2 ) v2.0
;			=0x01 -      SPI ( )
;			=0x02 -     SPI,   V1.0 (SD/MMC),    
;			=0x03 -     SPI,   V2.0 (SD/SDHC),    
;			=0x04 -     SPI,   V2.0,     SD-SDHC
;			=0x*5 -   ,      (  0- ,    )  ( 4  -  )
;			=0x*6 -  FAT16 - 0   
;			=0x*7 -  FAT16 -  2  FAT
;			=0x*8 -  FAT16 -   "FAT16"
;			=0x*9 -    
;			=0x*A -   
;			=0x*B -        FAT
;			=0x*C -
;			=0x*D -
;			=0x*E -
;			=0x*F -   ,  "FAT16" ,      "  " (,  )  

;-----------------------------------------------------------------------------
.SECTION
 SDCard_Run::
_SDCard_Run::
;-----------------------------------------------	
stage0:	;Initialization of SD
;-----------------------------------------------	
	lcall SDCard_InitCard

	cmp A,0
	jz stage1				;if success
	jmp return_code1		;if no card or FAT16
	

;-----------------------------------------------	
stage1:	;Find SubDir in RootDir
;-----------------------------------------------	

	; 6 
	;toLCD_String 0,0,"A=  "	;debug!!
	;mov A,[FClst+1]
	;call LCD_PrHexByte
	;mov A,[FClst+0]
	;call LCD_PrHexByte
	;mov A,[FSize+1]
	;call LCD_PrHexByte
	;mov A,[FSize+0]
	;call LCD_PrHexByte
	;halt                        ;


	;   [etap]=0x10, 0x20  0x40
	or [etap], 0x09					;  [etap]= 0x*9 ("   ")
	
	mov [ptr+0],<name1				;load name pointer
	mov [ptr+1],>name1
	lcall SDCard_FindFile_in_Root
	;   :
		;[etap] = 0x*0 - "", 0x*F - "  ", 0x*9 - "  " 
		;[FClst] (2) = subdir start cluster ( "")
		;[FSize] (4) = subdir size () ( "")
	tst [etap],0x0F					; 
	jnz return_code1				;0x*F - "  "  0x*9 - "  " 



	;mov [cnt],255
	;lcall SDCard_Skip
	;mov [cnt],255
	;lcall SDCard_Skip
	;toLCD_String 0,0,"E=     "	;debug!!
	;toLCD_Position  0, 2
	;mov A,[etap]
	;call LCD_PrHexByte
	;;toLCD_PrHexInt etap          ;  
	;toLCD_String 0,7,"cnt= "	;debug!!
	;mov A,[cntMax+1]
	;call LCD_PrHexByte
	;mov A,[cntMax+0]
	;call LCD_PrHexByte
	;halt
	
	; 6 
	;toLCD_String 0,0,"A="	;debug!!
	;mov A,[FClst+1]
	;call LCD_PrHexByte
	;mov A,[FClst+0]
	;call LCD_PrHexByte
	;mov A,[FSize+1]
	;call LCD_PrHexByte
	;mov A,[FSize+0]
	;call LCD_PrHexByte
	;halt                        ;

;-----------------------------------------------	
stage2:	;Find file in SubDir
;-----------------------------------------------	
	or [etap], 0x0A					;  [etap]= 0x*A ("   ")
	mov [ClstN+0],[FClst+0]			;load start cluster number (FClst   )
	mov [ClstN+1],[FClst+1]
	mov [ptr+0],<name2				;load name pointer
	mov [ptr+1],>name2

	;mov [dcnt],0					;    
	;toLCD_String 0,0,">"	;debug!!

.loop: ;    

	lcall SDCard_FindFile_in_Cluster	;   
	;   :
		;[etap] = 0x*0 - "", 0x*F - "  ", 0x*A - "   " 
		;[FClst] (2) = file start cluster ( "")
		;[FSize] (4) = file size          ( "")
	tst [etap],0x0F
	jz stage3						;=0 "  "

	tst [etap],0x01					; 0x*F - "  "  0x*A - "   "
	jnz return_code1				; "  "
	
	;     -   . 
	call SDCard_NextCluster			;  .   
	jnz return_code1				; A   0 - "  "	

	;  .   ""  "" FFF7..FFFF
	cmp [ClstN+1],0xFF
	jnz .loop			;,  < "FFF7" 
	cmp [ClstN+0],0xF7
	jc .loop			;,  < "FFF7" (  ! - ,        )
		;  ,      
		jmp return_code1			;[etap]=0x*A - "   "  (   )

;-----------------------------------------------	
;stage3:	;  
;-----------------------------------------------	
;	; 2 
;	toLCD_String 0,0,"File find="	;debug!!
;	mov A,[etap]
;	call LCD_PrHexByte


;	toLCD_String 1,0,"clst="	;debug!!
;	mov A,[FClst+1]
;	call LCD_PrHexByte
;	mov A,[FClst+0]
;	call LCD_PrHexByte
;	halt                          ;

;	; 2 
;	toLCD_String 0,0,"File find2="	;debug!!
;	mov A,[etap]
;	call LCD_PrHexByte
;	halt                          ;

;-----------------------------------------------	
stage3:	;Read Data
;-----------------------------------------------	
	;     - . 4  [etap]=0
	;mov [etap],0x70			;  
	call SDCard_ReadDataBuff		;   .   SDCard_Callback_Procedure!
	cmp A,0
	jz	stage4		;if success
	jmp return_code1
;-----------------------------------------------	
stage4:	;Data processings (if required)
;-----------------------------------------------	
	; 
	;toLCD_String 0,0,"process data    "	;debug!!
	;toLCD_PrHexInt etap          ;  
	;halt                        ;

	;    
;	toLCD_Position  0, 8    
	;toLCD_9_99 BSAddr ;Boot Sector address
;	toLCD_9_99 FatAddr ;Address of FAT area
;	toLCD_Position  1, 0    
;	toLCD_9_99 DirAddr ;Address of Dir area
;	toLCD_Position  1, 8    
;	toLCD_9_99 DataAddr ;Address of Data area
;	jmp t

	mov A,0	;return code
	ret	
return_code1:
	mov A,1
	ret	
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION: SDCard_Callback_Procedure
;       Callback procedure that is run from SDCard_ReadDataBuff
;  ARGUMENTS:
;       A - number of valid bytes
;       [buff] - data
;-----------------------------------------------------------------------------
.SECTION
 SDCard_Callback_Procedure::
_SDCard_Callback_Procedure::

	; -      16   

	;   
;	push A
;	toLCD_Position  0, 10    
;	pop A
;	lcall LCD_PrHexByte

	;  16    32 	
;	toLCD_Position  1, 0    
;	mov	X,buff+16			;   16 
.loop:
;    mov A,[X]
;    call LCD_WriteData      ;  
;	inc X
; 	mov A,X
;	cmp A, buff+16+16
;	jnz .loop
	
	ret
.ENDSECTION

;----------------------------------------------------------------------------------
;   -       
;           * -   ,
;           ? -  ,     ( )
;----------------------------------------------------------------------------------
.LITERAL
;   1   2   3   4   5   6   7   8     1   2   3  Attr: 0x00=file, 0x10=directory, 0x08=label
name1:
DB 'M','U','Z','D','A','T','A','*',  '*','*','*',0x10
;DB 'M','U','Z','*','*','*','*',' ',  '*','*','*',0x10
;DB 'D','C','I','M',' ',' ',' ',' ',  '*','*','*',0x10
;DB '0','0','1','L','a','m','b','a',  't','x','t',0x00

;   1   2   3   4   5   6   7   8     1   2   3  Attr
name2:
DB '?','?','?','*','*','*','*','*',  'b','o','r',0x00
;DB '1','2','9','*','*','*','*','*',  'b','o','r',0x00
;DB '2','2','3','*','*','*','*','*',  'b','o','r',0x00
;DB '*','*','*','N','*','*','*','*',  '*','*','*',0x10
;DB 'T','e','x','t','1',' ',' ',' ',  't','x','t',0x00
.ENDLITERAL


; End of File SDCard.asm
